/**
* \file: agwcl.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: sdc-ecm-engine
*
* \brief: Automotive Gateway Convenience Library
*
* \author: Kirill Marinushkin (kmarinushkin@de.adit-jv.com)
*
* \copyright (c) 2017 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
***********************************************************************/

#include <libgen.h>
#include <limits.h>
#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <openssl/asn1.h>
#include <openssl/bio.h>
#include <openssl/err.h>
#include <openssl/evp.h>
#include <openssl/pem.h>
#include <openssl/pkcs7.h>
#include <agwcl.h>
#include <sdc_keystore_keys.h>
#include <sdc_op_conv.h>
#include <sdc_perm.h>
#include <sdc_session.h>

#define CONF_PATH        "/etc/agwcl.conf"
#define CONF_P_PRIV_KEY  "wrapped_priv_key_path"
#define CONF_P_CLI_CERT  "wrapped_client_cert_path"
#define CONF_P_KEY_SIZE  "key_size"
#define CONF_P_KEY_ID    "key_id"

#define F_MASK_KEY_ALG    0x000F
#define F_MASK_HASH_ALG   0x00F0
#define F_MASK_FORMAT     0x0F00
#define F_MASK_ENCODING   0xF000
#define F_MASK_INPUT_ENC 0xF0000

enum params {
    PRIV_KEY = 0,
    CLIENT_CERT,
    KEY_SIZE,
    KEY_ID,
};

const char *params_sscanf[] = {
    [PRIV_KEY] = CONF_P_PRIV_KEY " = %s\n",
    [CLIENT_CERT] = CONF_P_CLI_CERT " = %s\n",
    [KEY_SIZE] = CONF_P_KEY_SIZE " = %u\n",
    [KEY_ID] = CONF_P_KEY_ID " = %u\n",
};

extern agwcl_error_t p7_sign_full(const EVP_MD *md, X509 *cert,
        unsigned char *data, size_t data_len,
        unsigned char **signature,  size_t *signature_len,
        uint64_t agwcl_flags);

static inline agwcl_error_t sdc_to_agwcl_error(sdc_error_t err) {
    return (agwcl_error_t)err;
}

static agwcl_error_t read_conf_value(enum params p, void *v)
{
    agwcl_error_t rc = AGWCL_ERR_CONF_PARAM_NOT_FOUND;
    FILE *f;
    ssize_t len;
    char *buf = NULL;
    size_t bufsize = 0;

    f = fopen(CONF_PATH, "r");
    if (!f)
        return AGWCL_ERR_CONF_FILE_NOT_ACCESSIBLE;

    while ((len = getline(&buf, &bufsize, f)) >= 0) {
        if (1 == sscanf(buf, params_sscanf[p], v)) {
            /* value found */
            rc = AGWCL_OK;
            goto out;
        }
    }

    /* value not found */
out:
    fclose(f);

    return rc;
}

static agwcl_error_t read_wrapped_file(enum params p, uint8_t **out_data,
        size_t *out_len)
{
    agwcl_error_t rc = AGWCL_ERR_UNKNOWN;
    FILE *f;
    struct stat st;
    char fpath[PATH_MAX] = "";

    rc = read_conf_value(p, fpath);
    if (rc)
        goto out;

    if (stat(fpath, &st)) {
        rc = AGWCL_ERR_WRAPPED_FILE_NOT_ACCESSIBLE;
        goto out;
    }

    *out_data = malloc(st.st_size);
    if (!*out_data) {
        rc = AGWCL_NO_MEM;
        goto out;
    }

    f = fopen(fpath, "r");
    if (!f) {
        rc = AGWCL_ERR_WRAPPED_FILE_NOT_ACCESSIBLE;
        goto out;
    }

    fread(*out_data, 1, st.st_size, f);
    fclose(f);

    *out_len = st.st_size;

    rc = AGWCL_OK;

out:

    return rc;
}

uint64_t get_flags_key_alg(uint64_t f)
{
    return (f & F_MASK_KEY_ALG);
}

uint64_t get_flags_hash_alg(uint64_t f)
{
    return (f & F_MASK_HASH_ALG);
}

uint64_t get_flags_format(uint64_t f)
{
    return (f & F_MASK_FORMAT);
}

uint64_t get_flags_encoding(uint64_t f)
{
    return (f & F_MASK_ENCODING);
}

uint64_t get_flags_input_encoding(uint64_t f)
{
    return (f & F_MASK_INPUT_ENC);
}

static agwcl_error_t parse_key_alg_sig_ver(uint64_t flags,
        sdc_sign_verify_alg_t *sdc_alg)
{
    switch (get_flags_key_alg(flags)) {
    case AGWCL_F_SIG_RSA:
        *sdc_alg = SDC_SIGNVER_ALG_RSA;
        break;
    default:
        return AGWCL_ERR_FLAG_KEY_ALG;
    }

    return AGWCL_OK;
}

static agwcl_error_t parse_key_alg_enc_dec(uint64_t flags,
        sdc_encrypt_decrypt_alg_t *sdc_alg)
{
    switch (get_flags_key_alg(flags)) {
    case AGWCL_F_DEC_RSA:
        *sdc_alg = SDC_ENCDEC_ALG_RSA;
        break;
    default:
        return AGWCL_ERR_FLAG_KEY_ALG;
    }

    return AGWCL_OK;
}

static agwcl_error_t parse_hash_alg_sig_ver(uint64_t flags,
        sdc_sign_verify_hash_t *sdc_hash)
{
    switch (get_flags_hash_alg(flags)) {
    case AGWCL_F_SIG_MD5:
        *sdc_hash = SDC_SIGNVER_HASH_MD5;
        break;
    case AGWCL_F_SIG_SHA1:
        *sdc_hash = SDC_SIGNVER_HASH_SHA1;
        break;
    case AGWCL_F_SIG_SHA224:
        *sdc_hash = SDC_SIGNVER_HASH_SHA224;
        break;
    case AGWCL_F_SIG_SHA256:
        *sdc_hash = SDC_SIGNVER_HASH_SHA256;
        break;
    case AGWCL_F_SIG_SHA384:
        *sdc_hash = SDC_SIGNVER_HASH_SHA384;
        break;
    case AGWCL_F_SIG_SHA512:
        *sdc_hash = SDC_SIGNVER_HASH_SHA512;
        break;
    default:
        return AGWCL_ERR_FLAG_HASH_ALG;
    }

    return AGWCL_OK;
}

static const EVP_MD *get_md_by_sdcid_sig_ver(sdc_sign_verify_hash_t sdc_hash)
{
    switch (sdc_hash) {
    case SDC_SIGNVER_HASH_MD5:
        return EVP_md5();
    case SDC_SIGNVER_HASH_SHA1:
        return EVP_sha1();
    case SDC_SIGNVER_HASH_SHA224:
        return EVP_sha224();
    case SDC_SIGNVER_HASH_SHA256:
        return EVP_sha256();
    case SDC_SIGNVER_HASH_SHA384:
        return EVP_sha384();
    case SDC_SIGNVER_HASH_SHA512:
        return EVP_sha512();
    default:
        return NULL;
    }
}

static agwcl_error_t parse_key_hash_alg_sig_ver(uint64_t flags,
        sdc_sign_verify_hash_t *sdc_hash, sdc_sign_verify_alg_t *sdc_alg)
{
    agwcl_error_t rc;

    rc = parse_key_alg_sig_ver(flags, sdc_alg);
    if (rc)
        goto out;

    rc = parse_hash_alg_sig_ver(flags, sdc_hash);
    if (rc)
        goto out;

    rc = AGWCL_OK;

out:
    return rc;
}

static sdc_error_t device_key_permissions_set(sdc_permissions_t *permissions)
{
    sdc_error_t rc;
    sdc_perm_bmsk_t allow = SDC_PERM_ENCRYPT | SDC_PERM_DECRYPT |
            SDC_PERM_SIGN | SDC_PERM_VERIFY |
            SDC_PERM_WRAP | SDC_PERM_UNWRAP;

    rc = sdc_set_default_permissions_current_gid(permissions);
    if (rc != SDC_OK)
        goto out;

    rc = sdc_permissions_owner_set(permissions, allow);
    if (rc != SDC_OK)
        goto out;

    rc = sdc_permissions_inherit_owner_set(permissions, SDC_PERM_NONE);
    if (rc != SDC_OK)
        goto out;

    rc = sdc_permissions_group_set(permissions, allow);
    if (rc != SDC_OK)
        goto out;

    rc = sdc_permissions_inherit_group_set(permissions, SDC_PERM_NONE);
    if (rc != SDC_OK)
        goto out;

    rc = sdc_permissions_others_set(permissions, SDC_PERM_NONE);
    if (rc != SDC_OK)
        goto out;

    rc = sdc_permissions_inherit_others_set(permissions, SDC_PERM_NONE);
    if (rc != SDC_OK)
        goto out;

    rc = SDC_OK;

out:
    return rc;
}

static agwcl_error_t load_key_force_import(sdc_session_t *session,
        sdc_key_id_t kid)
{
    agwcl_error_t rc = AGWCL_ERR_UNKNOWN;
    uint8_t *key_data = NULL;
    size_t key_data_len = 0;
    sdc_key_len_t keylen_val = SDC_KEY_LEN_UNKNOWN;
    size_t param_key_size = 0;
    sdc_permissions_t *permissions = NULL;
    pthread_mutex_t mut = PTHREAD_MUTEX_INITIALIZER;

    rc = sdc_to_agwcl_error(sdc_session_load_storage_key(session, kid));
    if (rc != sdc_to_agwcl_error(SDC_KID_NOT_AVAILABLE))
        return rc;

    /* not imported yet, need to import now */

    pthread_mutex_lock(&mut);

    /* prepare import */
    rc = read_wrapped_file(PRIV_KEY, &key_data, &key_data_len);
    if (rc)
        goto out;

    rc = read_conf_value(KEY_SIZE, &param_key_size);
    if (rc)
        goto out;

    rc = sdc_to_agwcl_error(
            sdc_import_formatted_autoload_key_with_secret_mod(
                    session,
                    key_data, key_data_len,
                    NULL, 0));
    if (rc)
        goto out;

    /* prepare permissions */
    rc = sdc_to_agwcl_error(sdc_permissions_alloc(&permissions));
    if (rc)
        goto out;

    rc = sdc_to_agwcl_error(device_key_permissions_set(permissions));
    if (rc)
        goto out;

    rc = sdc_to_agwcl_error(sdc_key_len_from_bits(param_key_size, &keylen_val));
    if (rc)
        goto out;

    /* do import */
    rc = sdc_to_agwcl_error(sdc_import_formatted_storage_key(
                    session,
                    SDC_KEY_FMT_RSA_PRIVATE, keylen_val, SDC_KEY_ENC_PEM,
                    &kid, SDC_CREATE_KEY_FIXED_ID, SDC_VOLATILE_STORAGE_KEY,
                    permissions, key_data, key_data_len));
    if (rc)
        goto out;

    /* load again */
    rc = sdc_to_agwcl_error(sdc_session_load_storage_key(session, kid));

out:
    pthread_mutex_unlock(&mut);

    sdc_permissions_free(permissions);

    return rc;
}

agwcl_error_t sign_with_device_key_pkcs1(unsigned char *data,
        size_t data_len, unsigned char **signature,  size_t *signature_len,
        uint64_t flags)
{
    agwcl_error_t rc = AGWCL_ERR_UNKNOWN;
    sdc_key_id_t kid = SDC_FLAG_INVALID_KID;
    sdc_sign_verify_hash_t sdc_hash = SDC_SIGNVER_HASH_INVALID;
    sdc_sign_verify_alg_t sdc_alg = SDC_SIGNVER_ALG_INVALID;
    sdc_sign_verify_type_t *sig_ver_type = NULL;
    sdc_sign_verify_desc_t *desc = NULL;
    sdc_session_t *session = NULL;
    uint8_t *iv = NULL;
    size_t iv_len = 0;

    if (get_flags_encoding(flags) != AGWCL_F_SIG_DER) {
        rc = AGWCL_ERR_FLAG_ENCODING;
        goto out;
    }

    rc = read_conf_value(KEY_ID, &kid);
    if (rc)
        goto out;

    rc = parse_key_hash_alg_sig_ver(flags, &sdc_hash, &sdc_alg);
    if (rc)
        goto out;

    rc = sdc_to_agwcl_error(sdc_sign_verify_type_alloc(&sig_ver_type));
    if (rc)
        goto out;

    rc = sdc_to_agwcl_error(sdc_sign_verify_type_set_alg(sig_ver_type, sdc_alg));
    if (rc)
        goto out;

    rc = sdc_to_agwcl_error(sdc_sign_verify_type_set_hash(sig_ver_type, sdc_hash));
    if (rc)
        goto out;

    switch (get_flags_input_encoding(flags)) {
    case AGWCL_F_SIG_PRECOMPUTED_HASH:
        rc = sdc_to_agwcl_error(sdc_sign_verify_type_set_opt_bmsk(sig_ver_type,
                                                                  SDC_SIGNVER_OPT_PRECOMP_HASH));
        if (rc)
            goto out;
        break;
    case AGWCL_F_SIG_MSG:
        /* nothing to do, this is the default of SDC */
        break;
    default:
        /* unexpected flag */
        rc = AGWCL_ERR_FLAG_INPUT_ENCODING;
        goto out;
    }

    /* load key into volatile storage */
    rc = sdc_to_agwcl_error(sdc_open_session(&session));
    if (rc)
        goto out;

    rc = load_key_force_import(session, kid);
    if (rc)
        goto out;

    /* get default signature size */
    rc = sdc_to_agwcl_error(sdc_sign_verify_desc_alloc(&desc));
    if (rc)
        goto out;

    rc = sdc_to_agwcl_error(sdc_sign_verify_desc_fill(session, sig_ver_type, desc));
    if (rc)
        goto out;

    rc = sdc_to_agwcl_error(sdc_sign_verify_desc_get_tag(desc, NULL, NULL, NULL, signature_len));
    if (rc)
        goto out;

    /* operation itself */
    rc = sdc_to_agwcl_error(sdc_sign(session, sig_ver_type,
                                     data, data_len,
                                     &iv, &iv_len,
                                     signature, signature_len));
    if (rc)
        goto out;

    rc = AGWCL_OK;

out:
    sdc_sign_verify_desc_free(desc);
    sdc_close_session(session);
    sdc_remove_storage_key(kid);
    sdc_sign_verify_type_free(sig_ver_type);

    return rc;
}

static agwcl_error_t sign_with_device_key_pkcs7(unsigned char *data,
        size_t data_len, unsigned char **signature,  size_t *signature_len,
        uint64_t flags)
{
    agwcl_error_t rc = AGWCL_ERR_UNKNOWN;
    uint8_t *cert_buf = NULL;
    size_t cert_buf_len = 0;
    BIO *cert_bio = NULL;
    X509 *cert_x = NULL;
    const EVP_MD *md = NULL;
    sdc_sign_verify_hash_t sdc_hash = SDC_SIGNVER_HASH_INVALID;
    sdc_sign_verify_alg_t sdc_alg = SDC_SIGNVER_ALG_INVALID;

    /* pkcs7 sign doesn't support precomputed hash */
    if (get_flags_input_encoding(flags) != AGWCL_F_SIG_MSG) {
        rc = AGWCL_ERR_FLAG_INPUT_ENCODING;
        goto out;
    }

    rc = parse_key_hash_alg_sig_ver(flags, &sdc_hash, &sdc_alg);
    if (rc)
        goto out;

    rc = get_client_cert(&cert_buf, &cert_buf_len, AGWCL_F_CERT_PEM);
    if (rc)
        goto out;

    cert_bio = BIO_new_mem_buf(cert_buf, cert_buf_len);
    if (!cert_bio) {
        rc = AGWCL_NO_MEM;
        goto out;
    }

    cert_x = PEM_read_bio_X509(cert_bio, NULL, NULL, NULL);
    if (!cert_x) {
        rc = AGWCL_ERR_CERT_FORMAT;
        goto out;
    }

    md = get_md_by_sdcid_sig_ver(sdc_hash);
    if (!md) {
        rc = AGWCL_ERR_FLAG_HASH_ALG;
        goto out;
    }

    rc = p7_sign_full(md, cert_x, data, data_len, signature,
                      signature_len, flags);
    if (rc)
        goto out;

    rc = AGWCL_OK;

out:
    BIO_free(cert_bio);
    free(cert_buf);

    return rc;
}

agwcl_error_t sign_with_device_key(unsigned char *data, size_t data_len,
                         unsigned char **signature,  size_t *signature_len,
                         uint64_t flags)
{
    switch (get_flags_format(flags)) {
    case AGWCL_F_SIG_PKCS1:
        return sign_with_device_key_pkcs1(data, data_len, signature,
                signature_len, flags);
    case AGWCL_F_SIG_PKCS7:
        return sign_with_device_key_pkcs7(data, data_len, signature,
                signature_len, flags);
    default:
        return AGWCL_ERR_FLAG_FORMAT;
    }
}

static agwcl_error_t decrypt_with_device_key_plain(unsigned char *cipherdata,
                            size_t cipherdata_len, unsigned char **plaintext,
                            size_t *plaintext_len, uint64_t flags)
{
    agwcl_error_t rc = AGWCL_ERR_UNKNOWN;
    sdc_key_id_t kid = SDC_FLAG_INVALID_KID;
    sdc_encrypt_decrypt_alg_t sdc_alg = SDC_ENCDEC_ALG_INVALID;
    sdc_encrypt_decrypt_type_t *enc_dec_type = NULL;
    sdc_encrypt_decrypt_desc_t *desc = NULL;
    sdc_session_t *session = NULL;
    uint8_t *iv = NULL;
    size_t iv_len = 0;

    if (get_flags_encoding(flags) != AGWCL_F_DEC_NOENC) {
        rc = AGWCL_ERR_FLAG_ENCODING;
        goto out;
    }

    rc = read_conf_value(KEY_ID, &kid);
    if (rc)
        goto out;

    rc = parse_key_alg_enc_dec(flags, &sdc_alg);
    if (rc)
        goto out;

    rc = sdc_to_agwcl_error(sdc_encrypt_decrypt_type_alloc(&enc_dec_type));
    if (rc)
        goto out;

    rc = sdc_to_agwcl_error(sdc_encrypt_decrypt_type_set_algorithm(enc_dec_type,
                                                                   sdc_alg));
    if (rc)
        goto out;

    rc = sdc_to_agwcl_error(sdc_encrypt_decrypt_type_set_block_mode(enc_dec_type,
                                                                    SDC_ENCDEC_BLK_NONE));
    if (rc)
        goto out;

    /* load key into volatile storage */
    rc = sdc_to_agwcl_error(sdc_open_session(&session));
    if (rc)
        goto out;

    rc = load_key_force_import(session, kid);
    if (rc)
        goto out;

    /* get default signature size */
    rc = sdc_to_agwcl_error(sdc_encrypt_decrypt_desc_alloc(&desc));
    if (rc)
        goto out;

    rc = sdc_to_agwcl_error(sdc_encrypt_decrypt_desc_fill(session, enc_dec_type, desc));
    if (rc)
        goto out;

    /* operation itself */
    rc = sdc_to_agwcl_error(sdc_decrypt(session, enc_dec_type,
                                        cipherdata, cipherdata_len,
                                        iv, iv_len,
                                        plaintext, plaintext_len));
    if (rc)
        goto out;

    rc = AGWCL_OK;

out:
    sdc_encrypt_decrypt_desc_free(desc);
    sdc_close_session(session);
    sdc_remove_storage_key(kid);
    sdc_encrypt_decrypt_type_free(enc_dec_type);

    return rc;
}

static agwcl_error_t decrypt_with_device_key_pkcs7(unsigned char *cipherdata,
                            size_t cipherdata_len, unsigned char **plaintext,
                            size_t *plaintext_len, uint64_t flags)
{
    agwcl_error_t rc = AGWCL_ERR_PKCS7_FORMAT;
    BIO *bin = NULL;
    PKCS7 *p7 = NULL;
    STACK_OF(PKCS7_RECIP_INFO) *rsk = NULL;
    X509_ALGOR *enc_alg = NULL;
    const EVP_CIPHER *evp_cipher = NULL;
    ASN1_OCTET_STRING *data_body = NULL;
    PKCS7_RECIP_INFO *ri = NULL;
    uint8_t *simk = NULL;
    size_t simklen = 0;
    uint8_t *rawk = NULL;
    size_t rawklen = 0;
    int tmplen = 0;
    int plaintext_len_s = 0;
    EVP_CIPHER_CTX ctx;
    uint64_t plain_flags = 0;

    bin = BIO_new_mem_buf(cipherdata, cipherdata_len);
    if (!bin) {
        rc = AGWCL_NO_MEM;
        goto out;
    }

    switch (get_flags_encoding(flags)) {
    case AGWCL_F_DEC_PLAIN:
        p7 = d2i_PKCS7_bio(bin, NULL);
        break;
    case AGWCL_F_DEC_SMIME:
        p7 = SMIME_read_PKCS7(bin, NULL);
        break;
    default:
        rc = AGWCL_ERR_FLAG_ENCODING;
    }

    if (!p7 || !PKCS7_type_is_enveloped(p7) ||
            OBJ_obj2nid(p7->type) != NID_pkcs7_enveloped)
        goto out;

    rsk = p7->d.enveloped->recipientinfo;
    if (!rsk)
        goto out;

    enc_alg = p7->d.enveloped->enc_data->algorithm;
    if (!enc_alg)
        goto out;

    evp_cipher = EVP_get_cipherbyobj(enc_alg->algorithm);
    if (!evp_cipher)
        goto out;

    data_body = p7->d.enveloped->enc_data->enc_data;
    if (!data_body)
        goto out;

    ri = sk_PKCS7_RECIP_INFO_value(rsk, 0);
    if (!ri)
        goto out;

    simk = ri->enc_key->data;
    simklen = ri->enc_key->length;
    if (!simk || !simklen)
        goto out;

    plain_flags = get_flags_key_alg(flags) | AGWCL_F_DEC_PLAIN |
            AGWCL_F_DEC_NOENC;

    rc = decrypt_with_device_key_plain(simk, simklen, &rawk, &rawklen,
            plain_flags);
    if (rc)
        goto out;

    *plaintext = realloc(*plaintext, data_body->length);
    if (!*plaintext) {
        rc = AGWCL_NO_MEM;
        goto out;
    }

    rc = AGWCL_ERR_DECRYPT;

    EVP_CIPHER_CTX_init(&ctx);

    if (1 != EVP_DecryptInit_ex(&ctx, evp_cipher, NULL, NULL, NULL))
        goto out;

    if (0 == EVP_CIPHER_asn1_to_param(&ctx, enc_alg->parameter))
        goto out;

    if (1 != EVP_DecryptInit_ex(&ctx, NULL, NULL, rawk, NULL))
        goto out;

    if (1 != EVP_DecryptUpdate(&ctx, *plaintext, &plaintext_len_s,
            data_body->data, data_body->length))
        goto out;

    if (plaintext_len_s < 0)
        goto out;

    if (1 != EVP_DecryptFinal_ex(&ctx, *plaintext + plaintext_len_s, &tmplen))
        goto out;

    if (tmplen < 0)
        goto out;

    plaintext_len_s += tmplen;

    *plaintext_len = plaintext_len_s;

    rc = AGWCL_OK;

out:
    free_secret_data(rawk, simklen);
    PKCS7_free(p7);
    BIO_free(bin);

    return rc;
}

agwcl_error_t decrypt_with_device_key(unsigned char *cipherdata,
                            size_t cipherdata_len, unsigned char **plaintext,
                            size_t *plaintext_len, uint64_t flags)
{
    switch (get_flags_format(flags)) {
    case AGWCL_F_DEC_PLAIN:
        return decrypt_with_device_key_plain(cipherdata, cipherdata_len,
                plaintext, plaintext_len, flags);
    case AGWCL_F_DEC_PKCS7:
        return decrypt_with_device_key_pkcs7(cipherdata, cipherdata_len,
                plaintext, plaintext_len, flags);
    default:
        return AGWCL_ERR_FLAG_FORMAT;
    }
}

agwcl_error_t get_client_cert(unsigned char **certificate,
        size_t *certificate_len, uint64_t flags)
{
    agwcl_error_t rc = AGWCL_ERR_UNKNOWN;
    uint8_t *wr_data = NULL;
    size_t wr_len = 0;
    size_t unwr_len = 0;
    sdc_session_t *session = NULL;
    sdc_wrap_unwrap_type_t *unwr_type = NULL;

    if (!certificate) {
        rc = AGWCL_INVALID_INPUT;
        goto out;
    }

    if (get_flags_encoding(flags) != AGWCL_F_CERT_PEM) {
        rc = AGWCL_ERR_FLAG_ENCODING;
        goto out;
    }

    rc = read_wrapped_file(CLIENT_CERT, &wr_data, &wr_len);
    if (rc)
        goto out;

    /* prepare unwrapper */
    rc = sdc_to_agwcl_error(sdc_open_session(&session));
    if (rc)
        goto out;

    rc = sdc_to_agwcl_error(sdc_unwrap_formatted_autoload_key_with_secret_mod(session,
                                                                              wr_data, wr_len,
                                                                              NULL, 0));
    if (rc)
        goto out;

    rc = sdc_to_agwcl_error(sdc_wrap_unwrap_type_alloc(&unwr_type));
    if (rc)
        goto out;

    rc = sdc_to_agwcl_error(sdc_unwrap_formatted_extract_type(wr_data, wr_len, &unwr_type));
    if (rc)
        goto out;

    /* operation itself */
    rc = sdc_to_agwcl_error(sdc_unwrap_formatted(session, unwr_type,
                                                 wr_data, wr_len,
                                                 certificate, &unwr_len));

    if (rc)
        goto out;

    rc = AGWCL_OK;

out:
    sdc_wrap_unwrap_type_free(unwr_type);
    sdc_close_session(session);

    if (certificate_len)
        *certificate_len = (rc ? 0 : unwr_len);

    return rc;
}

void free_secret_data(void *secret_data, size_t secret_data_len)
{
    if (secret_data != NULL)
        memset(secret_data, 0, secret_data_len);

    free(secret_data);
}
